  
  OPTION EXPLICIT
  OPTION DEFAULT NONE
  
  ' TO DO
  ' * Test over-voltage cut-out
  
  CONST CAL_pin = 4
  CONST REF_pin = 5
  CONST OVP_pin = 10
  CONST SER_pin = 16
  CONST SCK_pin = 17
  CONST RCK_pin = 18
  CONST GBAR_pin = 21
  CONST BOOST_pin = 22
  CONST CVFB_pin = 24
  CONST CFB_pin = 25
  CONST RELAY_ladder_out_to_buffer = &H1000
  CONST RELAY_out_gnd = &H2000
  CONST RELAY_ladder_in_from_Vref = &H4000
  CONST RELAY_out_terminals_to_current = &H8000
  CONST CURRENT_BIAS = 0.0006
  CONST MAX_Q1_TEMP = 100'120
  
  DIM FLOAT Rval = 12000
  DIM FLOAT Vref = 2.5'2.499  
  DIM FLOAT Rshunt = 0.1, volt_ranges(15), Q1temp = 25.0, Q1power, Q1joules, current_sink_calibration
  DIM INTEGER lastQ1check
  DIM INTEGER current_relay_states = 0, output_on = 0, thermal_protection = 0, thermal_protection_state
  DIM INTEGER pga_gain = 0
  DIM INTEGER current_mode = 3
  DIM INTEGER output_pulse_ms = 100
  DIM STRING screen_mode$ LENGTH 16, mode_from$ LENGTH 16, output_mode$ LENGTH 6 = "manual"
  DIM mode_names$(5) LENGTH 32
  DIM keyboard$ LENGTH 16
  DIM entry$ LENGTH 9
  DIM atten_mode$ LENGTH 3
  DIM protection_tripped$ LENGTH 8
  DIM FLOAT desired_voltage_etc, actual_voltage_range, ratio_left, ratio_right, resistance = 99999, actual_resistance = 99999
  DIM INTEGER actual_relay_setting, entry_set, i
  mode_names$(0) = "Voltage Divider"
  mode_names$(1) = "Voltage Divider (Buffered)"
  mode_names$(2) = "Voltage Reference"
  mode_names$(3) = "Voltage Reference (Buffered)"
  mode_names$(4) = "Current Reference"
  mode_names$(5) = "Resistance Reference"
  
  DIM STRING resdata(47) LENGTH 160
  resdata(0)="d42qqoQ2rquA2tqvw21qww1Vqxw1Vpy41U1zo2sq1g1Vl4Y2yq5I1Ut5o3Kq6o1qq7I2qp8I1Uq8o2Uq841Up+I2qm/I3Sq/o2kqAY5KrAo5UlDI5qpDg7LqEQ6r0FA7TqGI5KpGo6qlHI5SlHw6VpIg5TqJA5VV"
  resdata(1)="Jw5NrKo5alLQ6TqMI6qaMw5TlNg6tpOg5NpPY4srQI4yrQg6VmRg6VlSQ6rlTA6lmT460sUo5UVVQ5rlVg4rpWQ6zyXA4zrXw6jqYA7VkY44qqZw6llag5VZbY5SVcY6amdI6Umd45qVeo5spfQ5Vafo6slgQ61a"
  resdata(2)="hI6qih46kmiA6laiw6zmjo7UkkY6sako6MqlQ6rWmQ6panA6lknw5LVoo5UspQ5lVpo60lqQ6pkrA7Nmrw5NmsA6rUsw5ZVto6UauY5MVuo4sqvQ6lWwA7Lkww6lVxI5aZx45Ujyg6tUzQ7VWzg6rS0Q6pW1I6qc"
  resdata(3)="1w5pV2I5Kb246sV3o7KU4Y6ol4o6yW5Y5SW6I5aW645mV7I5sV7w61S8o4al9Y60U9g5Vk+Q41Z/I6SV/46mVAA9JVA4+sSBo+sZCQ+pZCg/TUDQ/NUEI/SiEw+TVFI+aiF4+qKGg9VNHQ9pZHg+ZVIQ+bWJI+kU"
  resdata(4)="Jw9LWKI+UUKw8rTLo/SSMQ9pWMo9WTNY+USOA+zSO49oVPA9LTPw9WZQo9kWRQ/LSRo+pVSY86VTA8zZT49UNUI9GVU48pVVo+YaWY9MTWo+qTXY8sTYI+qMY4/WSZI+WSZw9VRao9SSbY9SNbg+jVcQ+1KdI9KN"
  resdata(5)="d4+pSeA+ZSew9lSfg/ZWgQ9VKgo+2ShY+sKiA9TNiw/XUjI+YWjw+pKkg+rTlQ+nSlg+rMmQ83VnI+yKn4/KKoA+jZow81Npg/JSqQ9ZSqg9bSrY+aKsI9qKsw+tMtA+1Mt49qRuo9UKvY9kSvg8tNwQ9pUxI9mS"
  resdata(6)="x48aTyA+NUy4+qJzg9TL0Y/KM0g/LK1Q/aU2I9qL2w91S3I9WL349SK4g+lN5Y/SM5o+0M6Q+1R7A9rK74+MU8I+UN8w9rR9o9VN+Y+aN+g+3S/Y+URAJBSJAxCVOBJAyLBxDNMCpCaMDZBaJDpBYSERDWVFBAlL"
  resdata(7)="F5CUMGJDqVGxBtKHpDISIRDRSIhDaSJZBWJKBAzLK5CWNLBDZKLxClJMhDVONZCWRNpBsKOZDqKPBBVFP5CaOQJCKKQxBLJRhDVTSRBaNShBpLTRAVLUBCbMU5AqKVBBJLVxBbJWhCXKXRBqNXpBmLYRDFKZBArJ"
  resdata(8)="Z5BVKaBDWKaxA1KbpA2LcZAaLchCLKdZC0JeJCULexAtJfBDrKf5AUNgpCMKhRAlRhhBzNiZCWJjJBmJjxBSNkJCqIkxDWMlhDKKmRCjMmpA6LnRCtGoBBWJo5BcKpJBqFpxBzKqhC1GrZAWNrpBaFsZBGLtJCKM"
  resdata(9)="txDVGuBA7Nu5BZKvhApJwZAWLwpBJLxRA5LyJCoJy5BESzJA1Lz5BpL0pCxK1ZBkM1hBON2RDLJ3JBGR35AiL4BA2L45CaG5hCnJ6ZBlL6hA1F7RAjL8BDTJ8xA3L9BArF9xBZF+pBaG/ZA2J/pAmJAZGVJBBGzG"
  resdata(10)="B5HMJCBGyMCxGvKDhEtFEZE2KEpGUFFRG1FGBFTGGxGpFHJFlJHxEVJIhGrEJZGWGJpE0FKZGrKLJEcLLxHnKMJE6KMxHTGNpFxSOZHpKOpFVFPZHWIQBGTGQ5GyFRBFNGR5FSHShFXFTZFcMTpHUEUZHCKVBElF"
  resdata(11)="V5HMIWJHyKWxE1HXpGmIYZFFLYpHqIZRE5KaJGsEaxGbFbJFaHbxGzFchEnJdRGZFdpHFMeRHZIfJGZLfxE7JgBGVEg5FqHhpGKGiRHbIihHzKjRELLkBGjGk5EsHlJGqClxHLEmpEyHnRF1GnhGJJoZFZFpJHaE"
  resdata(12)="pxFqFqBFFFqxGXGrhHbGsRGROshFjFtZGmEuBGWGuxFlHvBGRJv5FyGwpGkExREzHxhHWIyREqFzJGUEzxFrH0JEaF05GMG1hFGK2RFqG2pFlF3RGdF4JHqE45EoF5BHFI55GZG6hEdJ7ZEtK7hHZE8ZGtG9JEqD"
  resdata(13)="9xGmF+JFZG+xG3G/hE5FARLrGApIkHBZK2ECBJJHC5KWEDJJEFD5JqEEhKVHFZLYGFhKaFGZJwKHBJ5JH5LaFIJJgiIxKZEJhKdIKZKsCKpKRLLRI7FMBLrEMxJ3JNJJsHNxI3FOhKpCPRLWEPpLdKQRJvLRJIiF"
  resdata(14)="R5I2GSBJHFS5K0CThIvJUZKiEUpLsJVRK5FWBKWIW5KtFXJLSCXxJhJYpJCOZRLZFZpLkEaRLjIbBJlDbxITFcBKmEc5JZHdpKmCeZKUCehJrDfRLtEgBKdEg5JRGhBL1Ih5JQFipKwLjZJqCjhKVDkZKMIlBIuF"
  resdata(15)="lxLdEmJJWEm5JFFnpKVEoZK5GohKxEpZJOHqJJUCqxLjGrJIxFr5JKEshImHtZKOGtpIuHuRIvOvJLEEv5KWCwJI1Gw5L0IxpITNyRKbCyhIXFzRKaE0JJCF0xKLE1JJkE1xJXD2pJmE3ZJSC3hKtD4ZK2C5BKTC"
  resdata(16)="55J6F6BJ2F6xIaF7pIQL8ZKaD8hLjE9ZJ6G+JJVI+xLmG/JLQI/5KGGAhOnCBRNTCBpOxECRPqEDJMuGD5MwFEBPJCExPnEFpNMEGZNLFGpOiCHROsFIBOiGIxOJQJJNxFJxNyGKhPOELRPaFLhNVBMZPVCNBNpC"
  resdata(17)="N5OlCOBNhFO5OmDPhPSFQROmCQpMrFRZPmFSBPWCSxPzGTJOMHT5MoDUpO5EVZNkCVhObDWZN8LXJNYEXxOXCYJOlPYxPGEZhNmIaRNPHahMfLbZNhFcBOWCc5OYCdBO2CdxOUFehOiFfZOKDfhO9EgZOrEhJPGC"
  resdata(18)="h5NODiBOLCi5OeFjpPiFkZMdJkhPDElZM8HmZNuHmpNlEnZNVCoZPdQpJNEDppOzFqZNyCrJOJQr5OhFsBNeGs5OpDthPdCuRMhKuhM1CvZPTEwBOuCw5PSPxJPICx5NaBypPcCzROyCzpPZC0RMaM1JNYC15OeI"
  resdata(19)="2JPOC2xMsM3pNoC4ZPoF4pORE5ZNiC6BPnF65PuG7BPqF7xNcG8hMrC9RNOD9pN5F+ZMtD/BNnC/5OuHABRrBA5RJQBhRaCCZS2DChQzEDRQtCEBScFExSmDFBQiFFxTHCGhRfFHZQyCHhTNDIZRhPJBRLBJ5SJE"
  resdata(20)="KJRlCK5QrGLhSMGMRRuIMpT2ENRRmCOBRbBOxQ+JPJRGEPxT1FQpToCRRQpERhTOCSRTqCTJRtETxS6CUBTnCUxSzBVpQdFWZQ2EWpShEXRReHYJTaHYxSRCZJR8FZ5RIEapRALbZTlCbhSUEcZSuDdJSrCd5S6D"
  resdata(21)="eJRsBexRfGfhTmCgZRmBghSSChhR5EiZT4QjBTyJjxSJDkhQlClRT2NlpS8CmZTwInJStDoJROCpJRGCpxROEqpQ2CrRR3EsBT7OsxTZHtJQyBtxQUFuhTkJvZSEEvhTvJwRTsGxBQqIx5QMDyBQFKyxQ5EzhR1B"
  resdata(22)="0RRCF0pQNH1RS3D2JRYB2xTCQ3BQkH3xSHE4hRJB5RScE5hTCG6RR7E7JRQE7xS0C8JQgo85TTC9pTxE+ZRyB+hQ3Q/RQ7EAJW7GAxVCMBBUlBBxX3ECpWODDZUoEDpWNCEZV1CFBVaBF5WqAGJXuLG5V6CHxVOC"
  resdata(23)="IpU9PJRVuCKJXIHKxWUHLBX+aL5WICMpWTHNZVnGNpWeCOZW+IPJWAUP5UzDQBV6IQ5UFSRpU1CSZVFESpUaETRXXDUBXYGU5WhCVhXJBWRUnEXJXDIXpVeEYZWQCZBU2CZxV3CaBVyCa5VQCb5UEOchVKBdZUaC"
  resdata(24)="eJVhDexU3CfJXIDgBWoEg5VGBhpX2FiRXbBjJVLIj5U6BkBXFBk5VJBlhXHfmZVOBmpUlQnRW1AoBUmIo5UgHpJXtCpxUVBqpUUIrZVCgrpU9HsRUJDtJUuCt5UUEuBWHCu5VFCvpVIBwRVGCwpW5DxZXhFyRVME"
  resdata(25)="zJU8IzxUjE0hU6E1ZVRB1hVkD2RWDf3JU5E3xU5B4JWTD45VBG5hU9E6RWHD6hXxC7RW8E8BXaD85VNB9JXBE95UWE+hUmC/RU8H/pW0AARZiCBJb1FB5bNDC5bcDDpb5JEZZ8EEhbhCFRa9DGBbyFGxYMFHJb5T"
  resdata(26)="HxbpDIxZOBJpZgQKRb6ELBb2FLxa/GMJYCeMxZHBNhY3BOZYSEOpYxCOxbODPpY5CQZYbGRJZhCR5agCSpZ5DTRYbBThafPUhbcCVZYuBWJZgDW5bgXXpZtBYZYcIZBbDDaJYAbbJb6Lbxa/Ic5YlBdpYwEeRYYF"
  resdata(27)="fJZbCfxYmBgJZAPgxaMChpakAiZaaAihZcCjZagHkBZ/XlBbIEl5bHEmpYtBnRbtBnhYfGoZZ1BpBbPfp5ZjCqJYcCqhZUBsJZFBspZTBtZYCKuBYyBu5YcgvBb9Gv5YYIwpb8NxRY0IyBYvCyxYTBzhbPD0JYYE"
  resdata(28)="05bRB1hbnH2RZ0C2paLD3RaEG4JYrE4xZ5B5hZ9C6RYdB7JZ7G7pYxB8RaDD9JbLH9xYXC+BYWE/BZ/v/5YZEAhcDcBZdCBCJfFBC5fgvD5f+KEhfGBFReBFFpeWAGRcaCHJfAEHxc/HIBcWCI5fAFJhdBDKZdXE"
  resdata(29)="KhdYELZe7CMBc9BMxfUDNhcBWOReZAPJfYAPpdgCQBe8CRJc8BR5ddBSBdoETBfECTxcXBUpfLDVZfXCWBfYCW5cYBXBeTAXxc8DYpcAdZZezBZhdhBaBdsBaxd7BcZeiAchdVAdRcWBeZcKEfRcaBgBe5AhJfyH"
  resdata(30)="h5cwBiBcNQjBf1PjpfyPkJelAk5cZBlxcNCmJfyDnxemApZd4BqBf/lrBe2As5dhBtpeuAuJcKByZeA+yhd/+zRf0FzpfzF05e3C1JdXC1xe4C2hcLE3Re7A3pcrB4RcGG5RcsB6JcMQ6xdcB7hfzP8Jf0D9ZecA"
  resdata(31)="9xfzD/JetAABj6CApj5CCBiXAC5iVADBjqAEJiKAExj0CFpiXDGJjHCHBj7vHxgHwIpiYAJpgzBKZhaALZgAyLxj/xNZi9PNxiNAPRiuAPhiCIQRiyASBhEISxjzfTZh5BURiaAVJh9PVphSAWZjdPXBhCEYJiDI"
  resdata(32)="YZi9HZJjdHZ5iZAaRiBQbhghQcxiCPdhh8PeZi+PfJhBIfpjePghghEhZg+IiJjeHixh/FjpiAFjxh/GkxiBIlhhTAmRi/CmphADnZjnCoJjACpJh+IppjdDqhh0BrZgBRsZj8FtJjuHtxh+PupgeQvRj9CvhjeH"
  resdata(33)="wpj8CxJj9Ix5hKAyphgBzRgSQ0BgkE05juD1pjbH1xgCP3Bj8O3Zh9E4Ri6A4hiCH5JgdE55iDE6pi9D7Bh8H8BgSC8ZjtD9hg+E+5jBHAJkhCBJl7EBhmvACRmEHDJkRIDxnhDFxlIEGhlNAHRkRBHhkkCIZmwA"
  resdata(34)="JBkeQJhniDKZl+ELBm9AL5m+DMpkjCNRm8DORkJIPJndBPxlLAQhl+ERJn2HR5mBHSpmvCTZk+CUBkJCU5kOQVJn2PWJmrAWxnxPXBkhBYRmsAapkOCbBnxDdxkeCeplMAgBk+CgpnBDhZl3EiRn+djRnEDkJkBe"
  resdata(35)="kZkRBmpkeBmxnhBnxlXAopnDDrRlWAr5kOBuhmRAvpn2fwRm4DxJnbByBldAy5kbIzhkEF0Zn7G0hkUE0xkCF1pn9G2ZlYA3Zn9E4BkwD45lnB5hkKE6RmCC7Rn8G8Bm0A85lcA9hkoE+pknE/RkKCABoUCAxryP"
  resdata(36)="BJogQBRoeBDpoTCERryDFJqAQFxp8DGhpCBHZo3EIJoJEIpp7CIxqEDJpobCKRpaAKhoOELRoKBMZrxHNRocCNpoJCOBr2DOxoBFPhrvPQBoBKQ5r+JRhqBCSRr+JTJrxDT5r+NUhqBDVRqICWBrvDW5qHCXhp/P"
  resdata(37)="YRoUBZBr/QZxr2Baxp+Cb5oTBcJoggcxp/gdppDBeJo+Bexo/gfxqiAgxp/fh5pBBiprAfkBoOBk5oIElpoGEmRqhAnRoFCoJrgDpBoPEpxrIDqxp4Crhr3PsRrkBtJrgftxoFBwBocBwZrjBxRo8ByJoGBypqJA"
  resdata(38)="zpoAf05qdA1JoIB3pqAE35qNA4poAi5Rr/I5pqhA6Rp/E7ZpAC8RorA9hr/O/JoAPApsASBZvADDRtRAD5sgBFxsPBGBstAHhtQCIRuOAJJsnBJZtHBKJvwfKhukALRuUALxv3ANBueAO5uTAPxsfBQJujAVBtSA"
  resdata(39)="VxtFAYBv/dY5sAebZtOAd5ssAi5sEEjJsEIkBsHIkxv7HlpsECmZv4HmptJAoJtGAoxsHCpZv4DqpsEQrZtRAsBv7BuRuPAyZtvCzBvIB0Bs4B2JubA45uEA5Bv7A6hucA7JtIAAxwpACZxNAGhyHALZwEgMJz4f"
  resdata(40)="Mxz/ENpyACORz/JOhwlAPxx/CQJyADSBz/NS5xABT5xEAUxx//VpwLIXBw/BXxwMEZJyA/Z5zzHapwAUbRz/+cJz9Lcpz7CdZwA/eBz8Lf5wLCgZzzDkxz8ClZx9BmRyfAm5wFEoxz6HpJz5HqpwFCrBz6Drxz/b"
  resdata(41)="s5znDthz4Ct5x7BupwFQvpz5PwBxHAxJwLBxRwMBzJwFBz5z5B1pwmA1xx8B2hxQB3Zy9A3hyoA4ZynA6Zz+F7RxUA8Bz+F+Jz+L+xxsA/Bz+CAZ1+BEp2FAFx1PAHB1+BKp2+AKx0qAMJ0kAO50vCPh3/7QB0YB"
  resdata(42)="RB14BR527ASR3+XU52LAVR2MAYB2+AYp0pAZR28Ac51dAfJ1eAh50oAo51FArB1GA2Z0CE2h0DE3p0CI4h0DC5J38D7R1eA8Z0CB8h39B9h0DBAp4CQBZ78PFZ6CAHx6DAJB4nAKJ4uALh4VAXp5CAaZ5vBbZ63A"
  resdata(43)="cB6YAcx64Adh7/3d54A4ep5bAg55LAiB5MAkx5cAnp4Cgn578fs54tA1p4lA6h4mA9Z4iAGB8uAHp8UAUB//FUh//CYB9/BYx//Lfx+/AnJ8WAwB9fAwR//XNKASARCAvAjiATAGKFfBGaGvAHqFXAISFYANKErA"
  resdata(44)="QCEsAViH/vV6EAwXiEXAjqEVAviEWACaMKAUCMEEVCMIEXaP7HXiMECYaMHEYqP7DbyP4HdSMICeKP3DfSMCEgiMEBhSP4DhqP9DiqP9HkiP8HlaP9BlyP8DtqMDBuSOCAvCMIBvaP3BwCPwDyaMHByiOEA4CMLA"
  resdata(45)="/qODABSRCATCQQBT6QPBUCSIAVCQBEWKQBEXKSHAXaQBCYCREAYqT+HZiT+HbqQBBbyT+BdSSBAkaRDAlyQBQmiT+PnyRBAoCQiA2aRBA9iQhAbKUhAgaUfBgiWQAhaWPAhiVIAkqVHAmSUkAqSURAx6UjA5iUSA"
  resdata(46)="oqYRAJycJALiigAMqhPANygoAOKgACPCj/DP6gnAQCj/BR6gAISSj/HUKiAAUigUAc6hAAdagTAjKgAQjSj/PuqggAvygKASqkQAT6kJAq6kAgrCn/feqoIAjioFAHa0EA4a5fA5K4vA8q4XAJ68LAALAFAubMCA"
  resdata(47)="UEZAAUUYgAVEYQAYkYIAl0YEAbkcCAAEsBAwN0AA"
  
FUNCTION FindNearestRes%(res%, actualreshere%)
  LOCAL INTEGER min, max, half, bits, actualres
  min = 0
  max = 1511
  DO
    half = (min+max) \ 2
    actualres = getresdata%(half, bits)
    IF res% < actualres THEN
      max = half - 1
      IF max < 0 THEN max = 0
    ELSE IF res% > actualres THEN
      min = half + 1
      IF min > 1511 THEN min = 1511
    ELSE
      max = half
    ENDIF
  LOOP UNTIL max - min < 2
  LOCAL INTEGER closestdelta = 99999, delta
  IF min < 1 THEN min = 1
  IF MAX > 1510 THEN max = 1510
  FOR half = min-1 TO max+1
    actualres = getresdata%(half, bits)
    delta = abs(res% - actualres)
    IF delta < closestdelta THEN
      closestdelta = delta
      actualreshere% = actualres
      FindNearestRes% = bits
    ENDIF
  NEXT half
END FUNCTION
  
SUB monitor_mosfet
  LOCAL FLOAT current, voltage, ref, vraw, Q1power
  current = (ShareADCwithSPIpin() + ShareADCwithSPIpin()) * 0.00478 * 0.5'PIN(CFB_pin)
  current = current * current - 0.04
  IF current < 0 THEN current = 0
  current = sqr(current)
  voltage = PIN(CVFB_pin)
  vraw = voltage
  ref = PIN(REF_pin)
  current = current * Vref / ref
  voltage = (voltage - current_sink_calibration) * 2148.62 / 47' + Vref
  IF voltage < 0 THEN voltage = 0
  Q1power = voltage * current
  Q1joules = Q1joules + Q1power * 0.01
  Q1temp = Q1temp - ((Q1temp - 25) - Q1power * 160) * 0.00025
  
  IF voltage >= 30 OR Q1power >= 8.3 OR Q1temp >= MAX_Q1_TEMP THEN
    thermal_protection = 1
    protection_tripped$ = "thermal"
    thermal_protection_state = current_relay_states
    update_relays 0
  ELSE IF thermal_protection > 0 and voltage <= 26 and Q1power <= 6 and q1temp <= 100 THEN
    thermal_protection = 0
    if current_relay_states = 0 THEN update_relays thermal_protection_state
  ENDIF
END SUB
  
sub mode_changed
  IF current_mode = 4 THEN
    SETTICK 10, monitor_mosfet, 1
    lastQ1check = TIMER
    Q1joules = 0
  ELSE
    SETTICK 0, 0, 1
  ENDIF
END SUB
  
SUB updateQ1
  LOCAL INTEGER now = TIMER
  if now > lastQ1check THEN
    LOCAL INTEGER time_passed = now - lastQ1check
    Q1power = Q1joules * 1000 / time_passed
    Q1joules = 0
    lastQ1check = now
  ENDIF
END SUB
  
  VAR RESTORE
  IF volt_ranges(1) = 0 THEN
    for i = 1 to 15
      volt_ranges(i) = (i+1)
    NEXT i
    '    volt_ranges(1) = 1.99993
    '    volt_ranges(2) = 2.99826
    '    volt_ranges(3) = 3.99818
    '    volt_ranges(4) = 5.00081
    '    volt_ranges(5) = 6.00073
    '    volt_ranges(6) = 6.99906
    '    volt_ranges(7) = 7.99899
    '    volt_ranges(8) = 9.02073
    '    volt_ranges(9) = 10.0207
    '    volt_ranges(10) = 11.019
    '    volt_ranges(11) = 12.0189
    '    volt_ranges(12) = 13.0215
    '    volt_ranges(13) = 14.0215
    '    volt_ranges(14) = 15.0198
    '    volt_ranges(15) = 16.0197
  ENDIF
  
SUB init_relays
  SETPIN SER_pin, DOUT, OC
  PIN(SER_pin) = 0
  SETPIN SCK_pin, DOUT, OC
  PIN(SCK_pin) = 0
  SETPIN RCK_pin, DOUT, OC
  PIN(RCK_pin) = 0
  SETPIN GBAR_pin, DOUT, OC
  PIN(GBAR_pin) = 1
END SUB
  
FUNCTION calc_ladder_resistance%(Rval%, value%)
  LOCAL INTEGER i
  LOCAL FLOAT ret
  ret = 1000000000
  for i = 0 TO 15
    if value% AND (1<<i) THEN ret = 1/(1/ret + 2/(Rval%*(4+i)))
  NEXT i
  calc_ladder_resistance% = ret + 0.5
END FUNCTION
  
  ' these two variables are manipulated in such a way that if update_relays is
  ' called in an interrupt handler *within update_relays*, the interrupt handler's
  ' value overrides that which is already being sent
  DIM INTEGER abort_update_relays, updating_relays
SUB update_relays states%
  LOCAL count%
  LOCAL INTEGER abort_update_relays_later
  
  IF thermal_protection > 0 THEN states% = 0
  
  IF current_relay_states <> states% THEN
    print "update_relays " + HEX$(states%)
    current_relay_states = states%
    
    IF updating_relays > 0 THEN abort_update_relays_later = 1
    updating_relays = 1
    PIN(SCK_pin) = 0
    PIN(RCK_pin) = 0
    FOR count% = 1 TO 32
      IF states% AND &H80000000 THEN PIN(SER_pin) = 1 ELSE PIN(SER_pin) = 0
      states% = states% << 1
      PIN(SCK_pin) = 1
      PIN(SCK_pin) = 0
    NEXT count%
    updating_relays = 0
    IF abort_update_relays = 1 THEN
      abort_update_relays = 0
    ELSE
      PIN(RCK_pin) = 1
      PIN(RCK_pin) = 0
      abort_update_relays = abort_update_relays_later
    ENDIF
    PIN(GBAR_pin) = 0
  ENDIF
END SUB
  
SUB set_voltage_range max_voltage!
  IF max_voltage! > Vref OR (current_mode = 1 AND output_on > 0) THEN PIN(BOOST_pin) = 1 ELSE PIN(BOOST_pin) = 0
  pga_gain = (max_voltage! / Vref) - 1
  LOCAL INTEGER new_value = (current_relay_states AND &HFFFFFF0F) OR (pga_gain << 4)
  update_relays new_value
END SUB
  
FUNCTION get_voltage_range!()
  LOCAL INTEGER pga_gain = ((current_relay_states >> 4) AND &HF) + 1
  IF pga_gain > 1 THEN
    get_voltage_range! = Vref * volt_ranges(pga_gain-1)
  ELSE
    get_voltage_range! = Vref * pga_gain
  ENDIF
END FUNCTION
  
FUNCTION get_voltage!()
  LOCAL INTEGER VALUE = current_relay_states >> 16
  VALUE = (VALUE>>8) + ((VALUE<<8) AND &HFF00)
  get_voltage! = get_voltage_range!() * VALUE / 65536
END FUNCTION
  
SUB set_voltage voltage!, buffered%, disconnect%, ignore_cal%, current_sink_mode%
  LOCAL FLOAT current_range
  IF ignore_cal% > 0 OR pga_gain = 0 THEN
    current_range = Vref * (pga_gain+1)
  ELSE
    current_range = Vref * volt_ranges(pga_gain)
  ENDIF
  LOCAL INTEGER VALUE = voltage! * 65535 / current_range
  IF VALUE >= 0 AND VALUE < 65536 THEN
    VALUE = (VALUE>>8) + ((VALUE<<8) AND &HFF00)
    LOCAL INTEGER new_value = (VALUE<<16) OR (pga_gain << 4)
    IF current_mode > 1 THEN new_value = new_value OR RELAY_ladder_in_from_Vref
    IF buffered% > 0 THEN
      IF disconnect% > 0 THEN
        new_value = new_value
      ELSE
        new_value = new_value OR RELAY_ladder_out_to_buffer OR RELAY_out_gnd
      ENDIF
    ELSE
      IF disconnect% = 0 and current_mode = 2 THEN
        new_value = new_value OR RELAY_out_gnd
      END IF
    END IF
    IF current_sink_mode% > 0 THEN new_value = new_value OR RELAY_ladder_out_to_buffer OR RELAY_out_gnd OR RELAY_out_terminals_to_current
    update_relays new_value
  END IF
END SUB
  
SUB set_voltage_all voltage!, buffered%, disconnect%, ignore_cal%, current_sink_mode%
  LOCAL INTEGER range
  LOCAL FLOAT factor
  IF ignore_cal% > 0 OR voltage! <= Vref THEN
    range = INT(voltage! / Vref) + 1
    IF range > 14 THEN range = 14
  ELSE
    range = 2
    DO WHILE range < 14 AND voltage! > Vref * volt_ranges(range-1)
      range = range + 1
    LOOP
  ENDIF
  set_voltage_range(range * Vref)
  IF disconnect% > 0 AND (buffered% = 0 OR current_mode = 1) THEN
    set_voltage 0, buffered%, disconnect%, ignore_cal%, current_sink_mode%
  ELSE
    set_voltage voltage!, buffered%, disconnect%, ignore_cal%, current_sink_mode%
  ENDIF
END SUB
  
SUB set_current current!
  LOCAL INTEGER VALUE = 65535 * current! / (Vref * 10) + 2
  IF VALUE >= 0 AND VALUE < 65536 THEN
    VALUE = (VALUE>>8) + ((VALUE<<8) AND &HFF00)
    LOCAL INTEGER new_value = (VALUE<<16) OR RELAY_ladder_in_from_Vref OR RELAY_ladder_out_to_buffer OR RELAY_out_gnd OR RELAY_out_terminals_to_current
    update_relays new_value
  END IF
END SUB
  
FUNCTION get_cal_delta!(num%)
  LOCAL count%, ret! = 0
  FOR count% = 1 to num%
    ret! = ret! + PIN(CAL_pin) - PIN(REF_pin)
    PAUSE 25
  NEXT count%
  get_cal_delta! = ret! / num%
END FUNCTION
  
SUB calibrate_range range!
  set_voltage_range range!
  LOCAL INTEGER VALUE = 65535 / (pga_gain+1), last_value
  LOCAL last_diff! = 999, this_diff!
  LOCAL count%, new_voltage!
  FOR count% = 1 TO 10
    new_voltage! = Vref * (pga_gain+1) * VALUE / 65535
    set_voltage new_voltage!, 1, 1, 1
    PAUSE 100
    this_diff! = get_cal_delta!(10)
    IF ABS(this_diff!) > ABS(last_diff!) THEN
      VALUE = last_value
      this_diff! = last_diff!
      count% = 10
    ELSE
      last_diff! = this_diff!
      last_value = VALUE
      IF this_diff! > 0 THEN
        IF this_diff! > 0.04 THEN
          IF this_diff! > 0.06 THEN
            VALUE = VALUE + 3
          ELSE
            VALUE = VALUE + 2
          ENDIF
        ELSE
          VALUE = VALUE + 1
        ENDIF
      ELSE
        IF this_diff! < -0.04 THEN
          IF this_diff! < -0.06 THEN
            VALUE = VALUE - 3
          ELSE
            VALUE = VALUE - 2
          ENDIF
        ELSE
          VALUE = VALUE - 1
        ENDIF
      ENDIF
    ENDIF
  NEXT count%
  volt_ranges(range! / Vref - 1) = 65535 / VALUE - this_diff! / (65 * Vref)
END SUB
  
SUB draw_circle x!, y!, rx!, ry!, sdeg!, fdeg!, degstep!, thick%, col%
  LOCAL FLOAT nx1, ny1, nx2, ny2, d
  LOCAL INTEGER i
  FOR i = 0 TO thick%-1
    FOR d = sdeg! to fdeg! - degstep! STEP degstep!
      nx1 = COS(RAD(d)) * (rx!-i/2)
      ny1 = SIN(RAD(d)) * (ry!-i/2)
      nx2 = COS(RAD(d + degstep!)) * (rx!-i/2)
      ny2 = SIN(RAD(d + degstep!)) * (ry!-i/2)
      LINE x! + nx1, y! + ny1, x! + nx2, y! + ny2, 1, col%
    NEXT d
  NEXT i
END SUB
  
SUB draw_omega x!, y!, w!, h!, t!, col%
  LINE x!, y! + h! - t! * 3, x! + w! / 4, y! + h! - t! * 3, t!, col%
  LINE x! + w!, y! + h! - t! * 3, x! + w! - w! / 4, y! + h! - t! * 3, t!, col%
  draw_circle x! + w! / 2, y! + h! / 2, w! / 2, w! / 2, 120, 410, (410-120)/8, 2, col%
END SUB
  
SUB draw_omega2 x!, y!, w!, h!, t!, col%
  LINE x!, y! + h! - t! * 3, x! + w! / 4, y! + h! - t! * 3, t!, col%
  LINE x! + w!, y! + h! - t! * 3, x! + w! - w! / 4, y! + h! - t! * 3, t!, col%
  draw_circle x! + w! / 2, y! + h! / 2, w! / 2, h! / 2, 125, 415, (415-125)/8, 1, col%
  draw_circle x! + w! / 2, y! + h! / 2, w! / 2 - 1, h! / 2 - 1, 125, 415, (415-125)/8, 1, col%
  draw_circle x! + w! / 2, y! + h! / 2, w! / 2 - 2, h! / 2 - 2, 125, 415, (415-125)/8, 1, col%
  draw_circle x! + w! / 2, y! + h! / 2, w! / 2 - 3, h! / 2 - 3, 125, 415, (415-125)/8, 1, col%
END SUB
  
  DIM INTEGER lfontw, lfonth, touchtimer, touchrepeat, touchaccel
  DIM STRING touchcmd
  
FUNCTION VOLTSTR8$(VAL!)
  LOCAL unit$ LENGTH 1
  IF current_mode = 4 THEN unit$ = "A" else unit$ = "V"
  IF VAL! <= 0.9999 THEN
    VOLTSTR8$ = STR$(VAL!*1000, 4, 1) + "m" + unit$
  ELSE
    VOLTSTR8$ = STR$(VAL!, 2, 3) + unit$ + " "
  ENDIF
END FUNCTIOn
  
FUNCTION VOLTSTR6$(VAL!)
  LOCAL unit$ LENGTH 1
  IF current_mode = 4 THEN unit$ = "A" else unit$ = "V"
  IF VAL! <= 0.0999 THEN
    VOLTSTR6$ = STR$(VAL!*1000, 2, 1) + "m" + unit$
  ELSEIF VAL! < 10.001 THEN
    VOLTSTR6$ = STR$(VAL!, 1, 3) + unit$
  ELSE
    VOLTSTR6$ = STR$(VAL!, 2, 2) + unit$
  ENDIF
END FUNCTIOn
  
FUNCTION RESSTR5$(VAL!)
  IF VAL! < 1000 THEN
    RESSTR5$ = STR$(VAL!, 5)
  ELSEIF VAL! < 10000 THEN
    RESSTR5$ = STR$(VAL!/1000, 1, 2) + "k"
  ELSEIF VAL! < 100000 THEN
    RESSTR5$ = STR$(VAL!/1000, 2, 1) + "k"
  ELSEIF VAL! < 1000000 THEN
    RESSTR5$ = STR$(VAL!/1000, 3, 0) + "k"
  ELSE
    RESSTR5$ = "high "
  ENDIF
END FUNCTION
  
FUNCTION RESSTR6$(VAL!)
  IF VAL! < 1000 THEN
    RESSTR6$ = STR$(VAL!, 6)
  ELSEIF VAL! < 10000 THEN
    RESSTR6$ = STR$(VAL!/1000, 2, 2) + "k"
  ELSEIF VAL! < 100000 THEN
    RESSTR6$ = STR$(VAL!/1000, 3, 1) + "k"
  ELSEIF VAL! < 1000000 THEN
    RESSTR6$ = STR$(VAL!/1000, 4, 0) + "k"
  ELSE
    RESSTR6$ = " high "
  ENDIF
END FUNCTION
  
FUNCTION RESSTR4$(VAL!)
  IF VAL! < 1000 THEN
    RESSTR4$ = STR$(VAL!, 4)
  ELSEIF VAL! < 10000 THEN
    RESSTR4$ = STR$(VAL!/1000, 1, 1) + "k"
  ELSEIF VAL! < 1000000 THEN
    RESSTR4$ = STR$(VAL!/1000, 3, 0) + "k"
  ELSE
    RESSTR4$ = "high"
  ENDIF
END FUNCTION
  
FUNCTION MSSTR$(VAL%)
  IF VAL% < 1000 THEN
    MSSTR$ = STR$(VAL%) + "ms"
  ELSEIF VAL% \ 1000 * 1000 = VAL% THEN
    MSSTR$ = STR$(VAL% \ 1000) + "s"
  ELSEIF VAL% < 10000 THEN
    MSSTR$ = STR$(VAL% / 1000.0, 1, 2) + "s"
  ELSE
    MSSTR$ = STR$(VAL% / 1000.0, 2, 1) + "s"
  ENDIF
END FUNCTION
  
FUNCTION DECSTR$(VAL!, CHARS%, ALIGN$)
  IF VAL! = INT(VAL!) THEN
    DECSTR$ = STR$(VAL!, CHARS%, 0)
  ELSEIF VAL! < 10 THEN
    DECSTR$ = STR$(VAL!, 1, CHARS%-2)
  ELSEIF CHARS% <= 2 THEN
    DECSTR$ = STR$(VAL!, 2, 0)
  ELSEIF VAL! < 100 THEN
    DECSTR$ = STR$(VAL!, 2, CHARS%-3)
  ELSEIF CHARS% <= 3 THEN
    DECSTR$ = STR$(VAL!, 3, 0)
  ELSEIF VAL! < 1000 THEN
    DECSTR$ = STR$(VAL!, 3, CHARS%-4)
  ELSEIF CHARS% <= 4 THEN
    DECSTR$ = STR$(VAL!, 4, 0)
  ELSEIF VAL! < 10000 THEN
    DECSTR$ = STR$(VAL!, 4, CHARS%-5)
  ELSEIF CHARS% <= 5 THEN
    DECSTR$ = STR$(VAL!, 5, 0)
  ELSEIF VAL! < 100000 THEN
    DECSTR$ = STR$(VAL!, 5, CHARS%-6)
  ELSEIF CHARS% <= 6 THEN
    DECSTR$ = STR$(VAL!, 6, 0)
  ELSEIF VAL! < 1000000 THEN
    DECSTR$ = STR$(6, CHARS%-7)
  ELSE
    DECSTR$ = STR$(VAL!)
  ENDIF
  IF ALIGN$ = "left" THEN
    WHILE LEFT$(DECSTR$, 1) = " "
      DECSTR$ = RIGHT$(DECSTR$, LEN(DECSTR$)-1) + " "
    WEND
  ENDIF
END FUNCTION
  
SUB drawQ1temp
  FONT 1, 2
  LOCAL powstr$ LENGTH 6
  updateQ1
  IF Q1power < 0.1 THEN
    powstr$ = "  0"
  ELSE
    powstr$ = DECSTR$(Q1power, 3)
  ENDIF
  TEXT MM.HRES*7/16, MM.VRES*13/16, powstr$ + "W / " + STR$(Q1temp, 3, 0) + "`C", CM, , , RGB(255,128,128), RGB(192,192,0)
END SUB
  
SUB update_output_state
  IF current_mode = 5 THEN
    LOCAL INTEGER VALUE = actual_relay_setting
    IF output_on = 0 THEN VALUE = 0
    IF VALUE >= 0 AND VALUE < 65536 THEN
      VALUE = (VALUE>>8) + ((VALUE<<8) AND &HFF00)
      update_relays VALUE << 16
    ENDIF
  ELSEIF current_mode = 4 THEN
    IF current_sink_calibration = 0 THEN
      LOCAL INTEGER i
      LOCAL FLOAT new_value
      set_voltage_all desired_voltage_etc + CURRENT_BIAS, 1, 1-output_on, 0, 0
      FOR i = 1 to 4
        current_sink_calibration = PIN(CVFB_pin)
      NEXT i
      current_sink_calibration = 0
      FOR i = 1 to 10
        current_sink_calibration = current_sink_calibration + PIN(CVFB_pin)
      NEXT i
      current_sink_calibration = current_sink_calibration / 10
    ENDIF
    IF output_on > 0 THEN
      set_voltage_all desired_voltage_etc + CURRENT_BIAS, 1, 1-output_on, 0, 1
    ELSE
      set_voltage_all CURRENT_BIAS, 1, 1-output_on, 0, 1
    ENDIF
  ELSE
    set_voltage_all desired_voltage_etc, (current_mode = 1 OR current_mode = 3), 1-output_on
  ENDIF
END SUB
  
SUB draw_desired
  LOCAL unit$ LENGTH 1
  IF current_mode = 4 THEN unit$ = "A" else unit$ = "V"
  FONT 8
  IF current_mode < 2 THEN
    IF atten_mode$ = ":" THEN
      TEXT MM.HRES*9/16, MM.VRES*9/32, DECSTR$(ratio_left, 4) + ":" + DECSTR$(ratio_right, 4, "left"), CM, , , RGB(WHITE), RGB(0,0,192)
    ELSEIF atten_mode$ = "1/x" THEN
      TEXT MM.HRES*9/16, MM.VRES*9/32, "1/" + STR$(ratio_right), CM, , , RGB(WHITE), RGB(0,0,192)
    ELSEIF atten_mode$ = "dB" THEN
      TEXT MM.HRES*9/16, MM.VRES*9/32, STR$(ratio_right) + "dB", CM, , , RGB(WHITE), RGB(0,0,192)
    ELSE
      TEXT MM.HRES*9/16, MM.VRES*9/32, STR$(ratio_left), CM, , , RGB(WHITE), RGB(0,0,192)
    ENDIF
  ELSEIF current_mode = 5 THEN
    IF resistance > 90000 THEN
      TEXT MM.HRES*9/16, MM.VRES*9/32, "-  high  +", CM, , , RGB(WHITE), RGB(0,0,192)
    ELSE
      TEXT MM.HRES*9/16, MM.VRES*9/32, "- " + RESSTR6$(resistance) + " +", CM, , , RGB(WHITE), RGB(0,0,192)
    ENDIF
    draw_omega2 MM.HRES*9/16+MM.FONTWIDTH*3, MM.VRES*9/32 - MM.FONTHEIGHT / 2 + 4, MM.FONTWIDTH, MM.FONTHEIGHT - 6, 2, RGB(WHITE)
  ELSE
    TEXT MM.HRES*9/16, MM.VRES*9/32, "-" + VOLTSTR8$(desired_voltage_etc) + "+", CM, , , RGB(WHITE), RGB(0,0,192)
  ENDIF
  update_output_state
  IF current_mode < 2 THEN
    TEXT MM.HRES*9/16, MM.VRES*19/32, "x" + STR$(desired_voltage_etc / Vref), CM, , , RGB(WHITE), RGB(192,0,192)
  ELSEIF current_mode = 5 THEN
    IF actual_resistance > 90000 THEN
      TEXT MM.HRES*9/16, MM.VRES*19/32, " high ", CM, , , RGB(WHITE), RGB(192,0,192)
    ELSE
      TEXT MM.HRES*9/16, MM.VRES*19/32, RESSTR6$(actual_resistance), CM, , , RGB(WHITE), RGB(192,0,192)
    ENDIF
    draw_omega2 MM.HRES*9/16+MM.FONTWIDTH*3, MM.VRES*19/32 - MM.FONTHEIGHT / 2 + 4, MM.FONTWIDTH, MM.FONTHEIGHT - 6, 2, RGB(WHITE)
  ELSE
    IF unit$ = "A" THEN
      LOCAL curvolt! = get_voltage!()
      TEXT MM.HRES*9/16, MM.VRES*19/32, VOLTSTR6$(curvolt! - CURRENT_BIAS) + " .1%", CM, , , RGB(WHITE), RGB(192,0,192)
    ELSE
      TEXT MM.HRES*9/16, MM.VRES*19/32, VOLTSTR6$(get_voltage!()) + " 2m" + unit$, CM, , , RGB(WHITE), RGB(192,0,192)
    ENDIF
    TEXT MM.HRES*9/16+lfontw, MM.VRES*19/32-lfonth/7, "+", LM, , , RGB(WHITE), RGB(192,0,192)
    LINE MM.HRES*9/16+lfontw+3, MM.VRES*19/32+lfonth/4, MM.HRES*9/16+lfontw*2-3, MM.VRES*19/32+lfonth/4, 3, RGB(WHITE)
  ENDIF
  LOCAL INTEGER input_resistance = calc_ladder_resistance%(Rval*1000, current_relay_states >> 16) / 1000
  FONT 1, 2
  IF current_mode = 4 THEN ' current reference mode
    drawQ1temp
  ELSEIF current_mode = 5 THEN
  ELSE
    TEXT MM.HRES*7/16, MM.VRES*13/16, "Zin=" + RESSTR5$(input_resistance) + " ", CM, , , RGB(WHITE), RGB(192,192,0)
    draw_omega MM.HRES*7/16 + MM.FONTWIDTH * 4, MM.VRES*13/16 - MM.FONTHEIGHT / 2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, RGB(WHITE)
  ENDIF
END SUB
  
FUNCTION get_output_resistance%(current_mode%)
  IF current_mode% = 1 OR current_mode% = 3 THEN
    get_output_resistance% = 0
  ELSEIF current_mode% = 0 OR current_mode% = 2 THEN
    get_output_resistance% = 2800
  ELSE
    get_output_resistance% = 1000000
  ENDIF
END FUNCTION
  
SUB draw_output_state
  FONT 1, 2
  BOX MM.HRES/8-1, MM.VRES*7/8, MM.HRES*5/8, MM.VRES/8, 1, RGB(WHITE), RGB(220,220,128)
  LOCAL INTEGER output_resistance = get_output_resistance%(current_mode)
  IF output_on = 0 THEN output_resistance = 1000000
  IF current_mode = 5 THEN
    IF output_on THEN
      TEXT MM.HRES*7/16, MM.VRES*15/16, "Output on", CM, , , RGB(WHITE), RGB(220,220,128)
    ELSE
      TEXT MM.HRES*7/16, MM.VRES*15/16, "Output off", CM, , , RGB(WHITE), RGB(220,220,128)
    ENDIF
  ELSE
    IF output_on = 0 AND (current_mode < 2 OR current_mode = 4) THEN
      TEXT MM.HRES*7/16, MM.VRES*15/16, "Zout= Off ", CM, , , RGB(WHITE), RGB(220,220,128)
    ELSE
      TEXT MM.HRES*7/16, MM.VRES*15/16, "Zout=" + RESSTR4$(output_resistance) + " ", CM, , , RGB(WHITE), RGB(220,220,128)
      draw_omega MM.HRES*7/16 + MM.FONTWIDTH * 4, MM.VRES*15/16 - MM.FONTHEIGHT / 2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, RGB(WHITE)
    ENDIF
  ENDIF
END SUB
  
SUB draw_screen
  FONT 1
  BOX 0, 0, MM.HRES/8, MM.VRES, 1, RGB(WHITE), RGB(192,0,0)
  TEXT MM.HRES/16, MM.VRES  /8, "+IN", CM, , , RGB(WHITE), RGB(192,0,0)
  LINE 0, MM.VRES/4, MM.HRES/8-1, MM.VRES/4, 1, RGB(WHITE)
  TEXT MM.HRES/16, MM.VRES*3/8, "-IN", CM, , , RGB(WHITE), RGB(192,0,0)
  LINE 0, MM.VRES/2, MM.HRES/8-1, MM.VRES/2, 1, RGB(WHITE)
  TEXT MM.HRES/16, MM.VRES*5/8, "+OUT", CM, , , RGB(WHITE), RGB(192,0,0)
  LINE 0, MM.VRES*3/4, MM.HRES/8-1, MM.VRES*3/4, 1, RGB(WHITE)
  TEXT MM.HRES/16, MM.VRES*7/8, "-OUT", CM, , , RGB(WHITE), RGB(192,0,0)
  
  BOX MM.HRES/8-1, 0, MM.HRES*7/8+1, MM.VRES/8, 1, RGB(WHITE), RGB(0,192,0)
  FONT 1
  TEXT MM.HRES*9/16, MM.VRES/16, "MODE: " + mode_names$(current_mode), CM, , , RGB(WHITE), RGB(0,192,0)
  BOX MM.HRES/8-1, MM.VRES/8-1, MM.HRES*7/8+1, MM.VRES*5/16+2, 1, RGB(WHITE), RGB(0,0,192)
  BOX MM.HRES/8-1, MM.VRES*7/16, MM.HRES*7/8+1, MM.VRES*5/16+1, 1, RGB(WHITE), RGB(192,0,192)
  FONT 8
  lfontw = MM.FONTWIDTH
  lfonth = MM.FONTHEIGHT
  FONT 1, 2
  BOX MM.HRES/8-1, MM.VRES*6/8, MM.HRES*5/8, MM.VRES/8 + 1, 1, RGB(WHITE), RGB(192,192,0)
  draw_desired
  draw_output_state
  BOX MM.HRES*3/4-2, MM.VRES*3/4, MM.HRES*1/4+2, MM.VRES/4, 1, RGB(WHITE), RGB(0,192,192)
  TEXT MM.HRES*7/8, MM.VRES*14/16, "Menu", CM, , , RGB(WHITE), RGB(0,192,192)
END SUB
  
SUB draw_calibration_menu
  BOX MM.HRES/8-1, 0, MM.HRES*7/8+1, MM.VRES, 1, RGB(WHITE), RGB(0,0,128)
  FONT 1, 2
  TEXT MM.HRES*9/16, MM.VRES*2/32, " Vref: " + STR$(Vref, 1, 6) + "V", CM, , , RGB(WHITE), RGB(0,0,128)
  TEXT MM.HRES*9/16, MM.VRES*5.5/32, "Shunt: " + STR$(Rshunt, 1, 5) + " ", CM, , , RGB(WHITE), RGB(0,0,128)
  draw_omega MM.HRES*9/16 + MM.FONTWIDTH * 6.5, MM.VRES*5.5/32 - MM.FONTHEIGHT / 2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, RGB(WHITE)
  TEXT MM.HRES*9/16, MM.VRES*9/32, "Rval: " + STR$(Rval, 5, 2) + " ", CM, , , RGB(WHITE), RGB(0,0,128)
  draw_omega MM.HRES*9/16 + MM.FONTWIDTH * 6.5, MM.VRES*9/32 - MM.FONTHEIGHT / 2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, RGB(WHITE)
  TEXT MM.HRES*9/16, MM.VRES*12.5/32, "Gain resistors:", CM, , , RGB(WHITE), RGB(0,0,128)
  LOCAL INTEGER i
  for i = 1 to 4
    TEXT MM.HRES*9/16, MM.VRES*(12.5+i*3)/32, STR$(Rval / (volt_ranges(2^(i-1)) - 1), 5, 2) + " ", CM, , , RGB(WHITE), RGB(0,0,128)
    draw_omega MM.HRES*9/16 + MM.FONTWIDTH * 3.5, MM.VRES*(12.5+i*3)/32 - MM.FONTHEIGHT / 2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, RGB(WHITE)
  NEXT i
  Box MM.HRes*2.5/16, MM.VRes*13.5/16, MM.HRes*15/32, MM.VRes/8, 1, RGB(WHITE), RGB(255,255,128)
  Text MM.HRes*12.5/32, MM.VRes*14.5/16, "Auto. Cal", CM, , , RGB(BLACK), RGB(255,255,128)
  Box MM.HRes*11.5/16, MM.VRes*13.5/16, MM.HRes*4/16, MM.VRes/8, 1, RGB(WHITE), RGB(255,255,128)
  Text MM.HRes*27/32, MM.VRes*14.5/16, "Back", CM, , , RGB(BLACK), RGB(255,255,128)
  screen_mode = "calib"
END SUB
  
SUB draw_main_menu
  LOCAL bgcol%
  BOX MM.HRES/8-1, 0, MM.HRES*7/8+1, MM.VRES, 1, RGB(WHITE), RGB(0,0,128)
  FONT 1, 2
  Box MM.HRes*4/16, MM.VRes*2/16, MM.HRes*10/16, MM.VRes/8, 1, RGB(WHITE), RGB(255,255,128)
  Text MM.HRes*9/16, MM.VRes*3/16, "Calibration", CM, , , RGB(BLACK), RGB(255,255,128)
  Text MM.HRes*9/16, MM.VRes*6/16, "Output:", CM, , , RGB(WHITE), RGB(0,0,128)
  IF output_mode$ = "manual" THEN bgcol% = RGB(255,255,128) ELSE bgcol% = RGB(224,224,96)
  Box MM.HRes*5/32, MM.VRes*15/32, MM.HRes*6/16, MM.VRes/8, 1, RGB(WHITE), bgcol%
  Text MM.HRes*11/32, MM.VRes*17/32, "Manual", CM, , , RGB(BLACK), bgcol%
  IF output_mode$ = "pulsed" THEN bgcol% = RGB(255,255,128) ELSE bgcol% = RGB(224,224,96)
  Box MM.HRes*19/32, MM.VRes*15/32, MM.HRes*6/16, MM.VRes/8, 1, RGB(WHITE), bgcol%
  Text MM.HRes*25/32, MM.VRes*17/32, "Pulsed", CM, , , RGB(BLACK), bgcol%
  Text MM.HRes*3/16, MM.VRes*11/16, "Duration:", LM, , , RGB(WHITE), RGB(0,0,128)
  Text MM.HRes*3/16, MM.VRes*13/16, MSSTR$(output_pulse_ms), LM, , , RGB(WHITE), RGB(0,0,128)
  Box MM.HRes*11.5/16, MM.VRes*13/16, MM.HRes*4/16, MM.VRes/8, 1, RGB(WHITE), RGB(255,255,128)
  Text MM.HRes*27/32, MM.VRes*14/16, "Back", CM, , , RGB(BLACK), RGB(255,255,128)
  screen_mode = "menu"
END SUB
  
SUB update_screen
  IF PIN(OVP_pin) = 0 and protection_tripped$ = "" THEN
'    output_on = 0
    update_relays 0
'    IF screen_mode$ = "" THEN draw_output_state
    protection_tripped$ = "voltage"
  ENDIF
  IF screen_mode = "" AND protection_tripped$ <> "" THEN
    BOX 8, MM.VRES/6, MM.HRES-16, MM.VRES*2/3, 1, RGB(WHITE), RGB(0,0,128)
    TEXT MM.HRES/2, MM.VRES*7/16, "output disabled", CM, , , RGB(WHITE), RGB(0,0,128)
    TEXT MM.HRES/2, MM.VRES*9/16, "due to excessive", CM, , , RGB(WHITE), RGB(0,0,128)
    IF protection_tripped$ = "thermal" THEN
      TEXT MM.HRES/2, MM.VRES*5/16, "Thermal protection", CM, , , RGB(RED), RGB(0,0,128)
      TEXT MM.HRES/2, MM.VRES*11/16, "temperature.", CM, , , RGB(WHITE), RGB(0,0,128)
    ELSE
      TEXT MM.HRES/2, MM.VRES*5/16, "Voltage protection", CM, , , RGB(RED), RGB(0,0,128)
      TEXT MM.HRES/2, MM.VRES*11/16, "voltage", CM, , , RGB(WHITE), RGB(0,0,128)
    ENDIF
    screen_mode$ = "warning"
    WHILE TOUCH(X) = -1
    WEND
    WHILE TOUCH(X) <> -1
    WEND
    PAUSE 250
    screen_mode$ = ""
    protection_tripped$ = ""
    draw_screen
  ELSEIF touchcmd = "dec_volt" THEN
    FONT 8
    IF TOUCH(X) = -1 THEN
      touchtimer = touchtimer - 1
    ELSEIF touchtimer > 0 THEN
      touchrepeat = touchrepeat + 1
    ENDIF
    
    IF touchtimer <= 0 THEN
      touchcmd = ""
      COLOUR RGB(WHITE)
    ELSE
      COLOUR RGB(255,0,0)
    ENDIF
    TEXT MM.HRES*9/16 - lfontw * 4.5, MM.VRES*9/32, "-", CM, , , , RGB(0,0,192)
    IF touchrepeat = 100 \ touchaccel THEN
      IF touchaccel < 100 THEN touchaccel = touchaccel + 1
      touchrepeat = 0
      IF touchaccel = 100 THEN
        inc_dec_voltage -100
      ELSEIF touchaccel >= 50 THEN
        inc_dec_voltage -10
      ELSE
        inc_dec_voltage -1
      ENDIF
      draw_desired
    END IF
  ELSEIF touchcmd = "inc_volt" THEN
    FONT 8
    IF TOUCH(X) = -1 THEN touchtimer = touchtimer - 1 ELSE IF touchtimer > 0 THEN touchrepeat = touchrepeat + 1
    IF touchtimer <= 0 THEN
      touchcmd = ""
      COLOUR RGB(WHITE)
    ELSE
      COLOUR RGB(255,0,0)
    ENDIF
    TEXT MM.HRES*9/16 + lfontw * 4.5, MM.VRES*9/32, "+", CM, , , , RGB(0,0,192)
    IF touchrepeat = 100 \ touchaccel THEN
      IF touchaccel < 100 THEN touchaccel = touchaccel + 1
      touchrepeat = 0
      IF touchaccel = 100 THEN
        inc_dec_voltage 100
      ELSEIF touchaccel >= 50 THEN
        inc_dec_voltage 10
      ELSE
        inc_dec_voltage 1
      ENDIF
      draw_desired
    END IF
  ELSEIF current_mode = 4 THEN
    IF screen_mode$ = "" and (TIMER - lastQ1check) >= 250 THEN drawQ1temp
  ENDIF
END SUB
  
SUB draw_mode_bar
  FONT 1
  BOX MM.HRES*1/4, MM.VRES/8-1, MM.HRES*3/4, 6*(MM.FONTHEIGHT*2), 1, RGB(white), RGB(0,224,0)
  LOCAL INTEGER i
  FOR i = 0 TO 5
    TEXT MM.HRES*10/16, MM.VRES/8-1+(i*2+1)*MM.FONTHEIGHT, mode_names$(i), CM, , , RGB(WHITE), RGB(0,224,0)
    IF i > 0 THEN LINE MM.HRES*1/4+MM.FONTWIDTH, MM.VRES/8-1+i*(MM.FONTHEIGHT*2), MM.HRES-MM.FONTWIDTH, MM.VRES/8-1+i*(MM.FONTHEIGHT*2), 1, RGB(WHITE)
  NEXT i
END SUB
  
SUB draw_kb_text x%, y%, text$, fg1%, fg2%, bg%
  TEXT x%, y%, text$, CM, , , fg1%, bg%
  TEXT x%-1, y%-1, text$, CM, , , fg2%, bg%
END SUB
  
SUB draw_value_entry kbd$, message$
  LOCAL INTEGER x, y, w, h, xp, yp, i
  LOCAL c$ LENGTH 1
  FONT 1, 2
  BOX MM.HRES/8-1, 0, MM.HRES*7/8+1, MM.VRES, 1, RGB(WHITE), RGB(0,0,224)
  BOX MM.HRES*3/16, MM.VRES/32, MM.HRES*6/8, MM.VRES/4, 1, RGB(WHITE), RGB(BLACK)
  FONT 8
  LINE MM.HRES*9/16 + MM.FONTWIDTH*3.5, MM.VRES*16/64, MM.HRES*9/16 + MM.FONTWIDTH*4.5, MM.VRES*16/64, 2, RGB(WHITE)
  IF message$ <> "" THEN
    FONT 1
    TEXT MM.HRES*7/32, MM.VRES*9/32, message$, LT, , , RGB(WHITE), RGB(0,0,224)
  ENDIF
  FONT 1, 2
  
  IF kbd$ = "" THEN
    IF current_mode < 2 THEN
      kbd$ = "789d456r123<.0:X"
    ELSE
      kbd$ = "789m456V123<.0cc"
    ENDIF
  ENDIF
  keyboard$ = kbd$
  
  entry$ = ""
  
  i = 0
  FOR y = 0 TO 3
    FOR x = 0 to 3
      i = i + 1
      c$ = MID$(keyboard$, i, 1)
      xp = MM.HRES*7/16-1+(MM.HRES/8)*x+3
      yp = MM.VRES*5/16-1+(MM.HRES/8)*y
      w = MM.HRES/9+2
      h = MM.HRES/9+2
      BOX xp, yp, w, h, 1, RGB(WHITE), RGB(160,160,255)
      IF c$ = "<" THEN
        TEXT xp-MM.FONTWIDTH/4+w/2, yp+h/2, "<", CM, , , RGB(BLACK), RGB(160,160,255)
        TEXT xp-MM.FONTWIDTH/4+w/2-1, yp+h/2-1, "<", CM, , , RGB(WHITE), RGB(160,160,255)
        LINE xp-MM.FONTWIDTH/2+w/2, yp+h/2-2, xp+MM.FONTWIDTH/2+w/2, yp+h/2-2, 1, RGB(WHITE)
        LINE xp-MM.FONTWIDTH/2+w/2, yp+h/2-1, xp+MM.FONTWIDTH/2+w/2, yp+h/2-1, 1, RGB(WHITE)
      ELSEIF c$ = "m" THEN
        draw_kb_text xp+2+w/2-1, yp+h/2, "mV", RGB(128,128,128), RGB(128,128,128), RGB(160,160,255)
      ELSEIF c$ = "a" THEN
        draw_kb_text xp+2+w/2-1, yp+h/2, "mA", RGB(128,128,128), RGB(128,128,128), RGB(160,160,255)
      ELSEIF c$ = "S" THEN
        draw_kb_text xp+2+w/2-1, yp+h/2, "ms", RGB(128,128,128), RGB(128,128,128), RGB(160,160,255)
      ELSEIF c$ = "M" THEN
        LOCAL STRING s$ LENGTH 1 = "m"
        IF entry_set > 0 THEN s$ = "k"
        TEXT xp+2+w/2-1, yp+h/2, s$ + " ", CM, , , RGB(128,128,128), RGB(160,160,255)
        draw_omega xp+2+w/2-1, yp+h/2 - MM.FONTHEIGHT/2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, RGB(128,128,128)
        TEXT xp+2+w/2-2, yp+h/2-1, s$ + " ", CM, , , RGB(128,128,128), RGB(160,160,255)
        draw_omega xp+2+w/2-2, yp+h/2-1 - MM.FONTHEIGHT/2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, RGB(128,128,128)
      ELSEIF c$ = "d" THEN
        draw_kb_text xp+2+w/2-1, yp+h/2, "dB", RGB(128,128,128), RGB(128,128,128), RGB(160,160,255)
      ELSEIF c$ = "c" THEN
        BOX xp, yp, w + MM.HRES/8, h, 1, RGB(WHITE), RGB(160,160,255)
        TEXT xp+w/2+MM.HRES/16, yp+h/2, "back", CM, , , RGB(BLACK), RGB(160,160,255)
        TEXT xp+w/2+MM.HRES/16-1, yp+h/2-1, "back", CM, , , RGB(WHITE), RGB(160,160,255)
        x = 3
      ELSEIF c$ = "V" THEN
        draw_kb_text xp+2+w/2, yp+h/2, "V", RGB(128,128,128), RGB(128,128,128), RGB(160,160,255)
      ELSEIF c$ = "A" THEN
        draw_kb_text xp+2+w/2, yp+h/2, "A", RGB(128,128,128), RGB(128,128,128), RGB(160,160,255)
      ELSEIF c$ = "O" THEN
        draw_omega xp+2+w/2 - MM.FONTWIDTH/2, yp+h/2 - MM.FONTHEIGHT/2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, RGB(128,128,128)
        draw_omega xp+2+w/2-1 - MM.FONTWIDTH/2, yp+h/2-1 - MM.FONTHEIGHT/2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, RGB(128,128,128)
      ELSEIF c$ = "r" THEN
        TEXT xp+2+w/2, yp+h/2, c$, CM, , , RGB(128,128,128), RGB(160,160,255)
        TEXT xp+2+w/2-1, yp+h/2-1, c$, CM, , , RGB(128,128,128), RGB(160,160,255)
      ELSE
        TEXT xp+2+w/2, yp+h/2, c$, CM, , , RGB(BLACK), RGB(160,160,255)
        TEXT xp+2+w/2-1, yp+h/2-1, c$, CM, , , RGB(WHITE), RGB(160,160,255)
      ENDIF
    NEXT x
  NEXT y
END SUB
  
SUB update_keyboard_buttons
  LOCAL INTEGER x, y, w, h, xp, yp, i, fg1, fg2
  LOCAL c$ LENGTH 1
  
  FONT 1, 2
  
  i = 0
  FOR y = 0 TO 3
    FOR x = 0 to 3
      i = i + 1
      c$ = MID$(keyboard$, i, 1)
      xp = MM.HRES*7/16-1+(MM.HRES/8)*x+3
      yp = MM.VRES*5/16-1+(MM.HRES/8)*y
      w = MM.HRES/9+2
      h = MM.HRES/9+2
      fg1 = RGB(BLACK)
      fg2 = RGB(WHITE)
      IF c$ = "m" THEN
        IF entry$ = "" OR VAL(entry$) > 37000 THEN
          fg1 = RGB(128,128,128)
          fg2 = RGB(128,128,128)
        ENDIF
        draw_kb_text xp+2+w/2-1, yp+h/2, "mV", fg1, fg2, RGB(160,160,255)
      ELSEIF c$ = "a" THEN
        IF entry$ = "" OR VAL(entry$) > 3000 THEN
          fg1 = RGB(128,128,128)
          fg2 = RGB(128,128,128)
        ENDIF
        draw_kb_text xp+2+w/2-1, yp+h/2, "mA", fg1, fg2, RGB(160,160,255)
      ELSEIF c$ = "M" THEN
        LOCAL STRING s$ LENGTH 1 = "m"
        IF entry_set > 0 THEN
          s$ = "k"
          IF mode_from$ = "calib" THEN
            IF VAL(entry$) < 0.001 OR VAL(entry$) > 100 THEN
              fg1 = RGB(128,128,128)
              fg2 = RGB(128,128,128)
            ENDIF
          ELSE
            IF VAL(entry$) < 1 OR VAL(entry$) > 1000 THEN
              fg1 = RGB(128,128,128)
              fg2 = RGB(128,128,128)
            ENDIF
          ENDIF
        ELSEIF VAL(entry$) <= 1 OR VAL(entry$) > 1000 THEN
          fg1 = RGB(128,128,128)
          fg2 = RGB(128,128,128)
        ENDIF
        TEXT xp+2+w/2-1, yp+h/2, s$ + " ", CM, , , fg1, RGB(160,160,255)
        draw_omega xp+2+w/2-1, yp+h/2 - MM.FONTHEIGHT/2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, fg1
        TEXT xp+2+w/2-2, yp+h/2-1, s$ + " ", CM, , , fg2, RGB(160,160,255)
        draw_omega xp+2+w/2-2, yp+h/2-1 - MM.FONTHEIGHT/2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, fg2
      ELSEIF c$ = "V" THEN
        IF entry$ = "" OR VAL(entry$) > 37 THEN
          fg1 = RGB(128,128,128)
          fg2 = RGB(128,128,128)
        ENDIF
        draw_kb_text xp+2+w/2, yp+h/2, "V", fg1, fg2, RGB(160,160,255)
      ELSEIF c$ = "A" THEN
        IF entry$ = "" OR VAL(entry$) > 3 THEN
          fg1 = RGB(128,128,128)
          fg2 = RGB(128,128,128)
        ENDIF
        draw_kb_text xp+2+w/2, yp+h/2, "A", fg1, fg2, RGB(160,160,255)
      ELSEIF c$ = "O" THEN
        IF entry_set > 0 THEN
          IF mode_from$ = "calib" THEN
            IF VAL(entry$) < 1 OR VAL(entry$) > 100000 THEN
              fg1 = RGB(128,128,128)
              fg2 = RGB(128,128,128)
            ENDIF
          ELSE
            IF VAL(entry$) < 1000 OR VAL(entry$) > 1000000 THEN
              fg1 = RGB(128,128,128)
              fg2 = RGB(128,128,128)
            ENDIF
          ENDIF
        ELSEIF VAL(entry$) <= 0 OR VAL(entry$) > 1 THEN
          fg1 = RGB(128,128,128)
          fg2 = RGB(128,128,128)
        ENDIF
        draw_omega xp+2+w/2 - MM.FONTWIDTH/2, yp+h/2 - MM.FONTHEIGHT/2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, fg1
        draw_omega xp+2+w/2-1 - MM.FONTWIDTH/2, yp+h/2-1 - MM.FONTHEIGHT/2, MM.FONTWIDTH, MM.FONTHEIGHT, 2, fg2
      ELSEIF c$ = "d" THEN
        IF INSTR(entry$, ":") OR entry$ = "" OR VAL(entry$) > 96.33 THEN
          fg1 = RGB(128,128,128)
          fg2 = RGB(128,128,128)
        ENDIF
        draw_kb_text xp+2+w/2-1, yp+h/2, "dB", fg1, fg2, RGB(160,160,255)
      ELSEIF c$ = "r" THEN
        LOCAL INTEGER ok = 0
        IF INSTR(entry$, ":") THEN
          LOCAL INTEGER colpos = INSTR(entry$, ":")
          IF colpos > 1 AND colpos < LEN(entry$) AND VAL(LEFT$(entry$, colpos-1)) > 0 AND VAL(RIGHT$(entry$, len(entry$)-colpos)) > 0 THEN ok = 1
        ELSE
          IF entry$ <> "" THEN ok = 1
          IF VAL(entry$) > 1 THEN ok = 2
        ENDIF
        IF ok = 2 THEN
          draw_kb_text xp+2+w/2-MM.FONTHEIGHT/3, yp+h/2-MM.FONTHEIGHT/6, "1", fg1, fg2, RGB(160,160,255)
          draw_kb_text xp+2+w/2+MM.FONTHEIGHT/4, yp+h/2+MM.FONTHEIGHT/6, c$, fg1, fg2, RGB(160,160,255)
          LINE xp+2+w/2-MM.FONTHEIGHT/3, yp+h/2+MM.FONTHEIGHT/3, xp+2+w/2+MM.FONTHEIGHT/6, yp+h/2-MM.FONTHEIGHT/3, 1, RGB(WHITE)
          LINE xp+2+w/2-MM.FONTHEIGHT/3, yp+h/2+MM.FONTHEIGHT/3-1, xp+2+w/2+MM.FONTHEIGHT/6, yp+h/2-MM.FONTHEIGHT/3-1, 1, RGB(WHITE)
        ELSE
          IF NOT ok THEN
            fg1 = RGB(128,128,128)
            fg2 = RGB(128,128,128)
          ENDIF
          BOX xp, yp, w, h, 1, RGB(WHITE), RGB(160,160,255)
          draw_kb_text xp+2+w/2, yp+h/2, c$, fg1, fg2, RGB(160,160,255)
        ENDIF
      ELSEIF c$ = "s" THEN
        IF VAL(entry$) < 0.01 OR VAL(entry$) > 10 THEN
          fg1 = RGB(128,128,128)
          fg2 = RGB(128,128,128)
        ENDIF
        draw_kb_text xp+2+w/2-1, yp+h/2, "s", fg1, fg2, RGB(160,160,255)
      ELSEIF c$ = "S" THEN
        IF VAL(entry$) < 10 OR VAL(entry$) > 10000 THEN
          fg1 = RGB(128,128,128)
          fg2 = RGB(128,128,128)
        ENDIF
        draw_kb_text xp+2+w/2-1, yp+h/2, "ms", fg1, fg2, RGB(160,160,255)
      ENDIF
    NEXT x
  NEXT y
END SUB
  
SUB inc_dec_voltage by!
  IF current_mode = 5 THEN
    LOCAL INTEGER rint
    resistance = resistance + by! * 10
    actual_relay_setting = FindNearestRes%(resistance, rint)
    actual_resistance = rint
  ELSE
    IF desired_voltage_etc < 1 THEN
      desired_voltage_etc = desired_voltage_etc + by! / 10000
    ELSEIF desired_voltage_etc < 10 THEN
      desired_voltage_etc = desired_voltage_etc + by! / 1000
    ELSE
      desired_voltage_etc = desired_voltage_etc + by! / 100
    END IF
    IF desired_voltage_etc < 0 THEN desired_voltage_etc = 0
    IF desired_voltage_etc > 37 THEN desired_voltage_etc = 37
  ENDIF
END SUB
  
SUB output_pulse_off
  SETTICK 0, output_pulse_off, 2
  IF output_on > 0 AND output_mode$ = "pulsed" THEN
    output_on = 0
    update_output_state
    IF screen_mode$ = "" THEN draw_output_state
  ENDIF
END SUB
  
SUB TouchInt
  LOCAL INTEGER tx = TOUCH(X), ty = TOUCH(Y)
  
  IF screen_mode$ = "mode_bar" THEN
    FONT 1
    screen_mode$ = ""
    IF tx >= MM.HRES*1/4 AND ty >= MM.VRES/8-1 AND ty < MM.VRES/8-1+6*(MM.FONTHEIGHT*2) THEN
      current_mode = (ty-(MM.VRES/8-1)) \ (MM.FONTHEIGHT*2)
      mode_changed
      VAR SAVE current_mode
    ENDIF
    draw_screen
  ELSEIF screen_mode$ = "keyboard" THEN
    LOCAL INTEGER x, y, w, h, xp, yp, i
    LOCAL c$ LENGTH 1
    i = 0
    FOR y = 0 TO 3
      FOR x = 0 to 3
        i = i + 1
        c$ = MID$(keyboard$, i, 1)
        xp = MM.HRES*7/16-1+(MM.HRES/8)*x+3
        yp = MM.VRES*5/16-1+(MM.HRES/8)*y
        w = MM.HRES/9+2
        h = MM.HRES/9+2
        IF tx >= xp AND ty >= yp and tx <= xp + w and ty <= yp + h THEN
          LOCAL draw$ LENGTH 10
          IF c$ = "c" or c$ = "X" THEN
            IF mode_from$ = "menu" THEN
              draw_main_menu
            ELSEIF mode_from$ = "calib" THEN
              draw_calibration_menu
            ELSE
              screen_mode$ = ""
              draw_screen
            ENDIF
          ELSEIF c$ = "<" THEN
            IF LEN(entry$) > 0 THEN entry$ = LEFT$(entry$, LEN(entry$)-1)
            draw$ = " " + entry$
          ELSEIF c$ = "m" THEN
            IF entry$ <> "" and VAL(entry$) <= 37000 THEN
              IF mode_from$ = "calib" THEN
                Vref = VAL(entry$) / 1000.0
                VAR SAVE Vref
                draw_calibration_menu
              ELSE
                desired_voltage_etc = VAL(entry$) / 1000.0
                screen_mode$ = ""
                draw_screen
              ENDIF
            ENDIF
          ELSEIF c$ = "a" THEN
            IF entry$ <> "" and VAL(entry$) <= 3000 THEN
              desired_voltage_etc = VAL(entry$) / 1000.0
              screen_mode$ = ""
              draw_screen
            ENDIF
          ELSEIF c$ = "M" THEN
            IF mode_from$ = "calib" THEN
              IF entry_set = 0 THEN
                IF VAL(entry$) >= 1 and VAL(entry$) <= 1000 THEN
                  Rshunt = VAL(entry$) / 1000.0
                  VAR SAVE Rshunt
                  draw_calibration_menu
                ENDIF
              ELSEIF VAL(entry$) >= 0.001 AND VAL(entry$) < 100 THEN
                volt_ranges(1 << (entry_set-1)) = (12 / VAL(entry$) + 1)
                VAR SAVE volt_ranges()
                draw_calibration_menu
              ENDIF
            ELSE
              IF VAL(entry$) >= 1 and VAL(entry$) <= 100 THEN
                LOCAL INTEGER rvalint, rint
                resistance = VAL(entry$)*1000
                actual_relay_setting = FindNearestRes%(resistance, rint)
                actual_resistance = rint
                screen_mode$ = ""
                draw_screen
              ENDIF
            ENDIF
          ELSEIF c$ = "O" THEN
            IF mode_from$ = "calib" THEN
              IF entry_set = 0 THEN
                IF VAL(entry$) > 0 and VAL(entry$) <= 1 THEN
                  Rshunt = VAL(entry$)
                  VAR SAVE Rshunt
                  draw_calibration_menu
                ENDIF
              ELSEIF VAL(entry$) >= 1 AND VAL(entry$) <= 100000 THEN
                volt_ranges((1 << entry_set) - 1) = (Rval / VAL(entry$) + 1)
                VAR SAVE volt_ranges()
                draw_calibration_menu
              ENDIF
            ELSE
              IF VAL(entry$) >= 1000 and VAL(entry$) <= 100000 THEN
                LOCAL INTEGER rint
                resistance = VAL(entry$)
                actual_relay_setting = FindNearestRes%(resistance, rint)
                actual_resistance = rint
                screen_mode$ = ""
                draw_screen
              ENDIF
            ENDIF
          ELSEIF c$ = "V" THEN
            IF entry$ <> "" AND VAL(entry$) <= 37 THEN
              IF mode_from$ = "calib" THEN
                Vref = VAL(entry$)
                VAR SAVE Vref
                draw_calibration_menu
              ELSE
                desired_voltage_etc = VAL(entry$)
                screen_mode$ = ""
                draw_screen
              ENDIF
            ENDIF
          ELSEIF c$ = "A" THEN
            IF entry$ <> "" AND VAL(entry$) <= 3 THEN
              desired_voltage_etc = VAL(entry$)
              screen_mode$ = ""
              draw_screen
            ENDIF
          ELSEIF c$ = "d" THEN
            IF NOT INSTR(entry$, ":") and entry$ <> "" AND VAL(entry$) <= 96.33 THEN
              atten_mode$ = "dB"
              ratio_right = VAL(entry$)
              ratio_left = 1 / (10 ^ (ratio_right / 20))
              desired_voltage_etc = Vref * ratio_left
              screen_mode$ = ""
              draw_screen
            ENDIF
          ELSEIF c$ = "r" THEN
            IF INSTR(entry$, ":") THEN
              LOCAL INTEGER colpos = INSTR(entry$, ":")
              IF colpos > 1 AND colpos < LEN(entry$) THEN
                LOCAL FLOAT leftval = VAL(left$(entry$, colpos-1))
                LOCAL FLOAT rightval = VAL(RIGHT$(entry$, len(entry$)-colpos))
                IF leftval > 0 AND rightval > 0 THEN
                  atten_mode$ = ":"
                  ratio_left = leftval
                  ratio_right = rightval
                  desired_voltage_etc = Vref * leftval / (leftval + rightval)
                  screen_mode$ = ""
                  draw_screen
                ENDIF
              ENDIF
            ELSE
              IF entry$ <> "" THEN
                LOCAL FLOAT ratio = VAL(entry$)
                IF ratio > 1 THEN
                  atten_mode$ = "1/x"
                  ratio_right = ratio
                  ratio_left = 1 / ratio
                  desired_voltage_etc = Vref * ratio_left
                ELSE
                  atten_mode$ = "x"
                  ratio_left = ratio
                  desired_voltage_etc = Vref * ratio_left
                ENDIF
                screen_mode$ = ""
                draw_screen
              ENDIF
            ENDIF
          ELSEIF c$ = "s" THEN
            output_pulse_ms = VAL(entry$) * 1000
            VAR SAVE output_pulse_ms
            draw_main_menu
          ELSEIF c$ = "S" THEN
            output_pulse_ms = VAL(entry$)
            VAR SAVE output_pulse_ms
            draw_main_menu
          ELSEIF LEN(entry$) < 9 THEN
            IF c$ = ":" THEN
              IF NOT INSTR(entry$, c$) THEN entry$ = entry$ + c$
            ELSEIF c$ = "." THEN
              IF INSTR(entry$, ":") THEN
                LOCAL INTEGER colpos = INSTR(entry$, ":")
                IF NOT INSTR(colpos, entry$, c$) THEN entry$ = entry$ + c$
              ELSE
                IF NOT INSTR(entry$, c$) THEN entry$ = entry$ + c$
              ENDIF
            ELSE
              entry$ = entry$ + c$
            ENDIF
            draw$ = entry$
          ENDIF
          IF draw$ <> "" THEN
            FONT 8
            TEXT MM.HRES*9/16 + MM.FONTWIDTH * 4.5, MM.VRES*10/64, draw$, RM, , , RGB(WHITE), RGB(BLACK)
            LINE MM.HRES*9/16 + MM.FONTWIDTH*3.5, MM.VRES*16/64, MM.HRES*9/16 + MM.FONTWIDTH*4.5, MM.VRES*16/64, 2, RGB(WHITE)
            update_keyboard_buttons
          ENDIF
        END IF
      NEXT x
    NEXT y
  ELSEIF screen_mode$ = "menu" THEN
    IF tx > MM.HRES/8 THEN
      IF ty >= MM.VRES*1/16 and ty <= MM.VRES*3/16 THEN 'calibration button
        draw_calibration_menu
      ELSEIF ty >= MM.VRES*15/32 and ty <= MM.VRES*19/32 THEN
        IF tx < MM.HRES*9/16 THEN 'manual output
          output_mode$ = "manual"
          VAR SAVE output_mode$
          draw_main_menu
        ELSE 'pulsed output
          output_mode$ = "pulsed"
          VAR SAVE output_mode$
          draw_main_menu
        ENDIF
      ELSEIF ty >= MM.VRES*12/16 and ty <= MM.VRES*14/16 and tx < MM.HRES*5/8 THEN 'set pulse duration
        draw_value_entry "789s456S123<.0cc", "Pulse ms"
        mode_from$ = screen_mode$
        screen_mode$ = "keyboard"
      ELSEIF ty >= MM.VRES*13/16 and ty <= MM.VRES*15/16 AND tx > MM.HRES*5/8 THEN 'back
        screen_mode$ = ""
        draw_screen
      ENDIF
    ENDIF
  ELSEIF screen_mode$ = "calib" THEN
    If ty >= MM.VRes*1/32 And ty <= MM.VRes*5/32 Then
      draw_value_entry "789m456V123<.0cc", "New Vref"
      mode_from$ = screen_mode$
      screen_mode$ = "keyboard"
    ELSEIF ty >= MM.VRes*5/32 and ty <= MM.VRes*9/32 THEN
      draw_value_entry "789M456O123<.0cc", "New Rshunt"
      mode_from$ = screen_mode$
      screen_mode$ = "keyboard"
      entry_set = 0
    ELSEIF ty >= MM.VRES*9/32 AND ty <= MM.VRES*25/32 THEN
      LOCAL INTEGER which = (ty / MM.VRES - 14/32) * 32 / 3
      draw_value_entry "789M456O123<.0cc", "New " + str$(12 / (2^which)) + "k"
      mode_from$ = screen_mode$
      screen_mode$ = "keyboard"
      entry_set = which+1
    ELSEIF ty >= MM.VRES*25/32 THEN
      IF tx < MM.HRES*3/4 THEN
        calibrate_range 5
        draw_calibration_menu
        calibrate_range 7.5
        draw_calibration_menu
        calibrate_range 12.5
        draw_calibration_menu
        calibrate_range 22.5
        set_voltage_range Vref
        draw_calibration_menu
        LOCAL INTEGER i
        LOCAL FLOAT res
        FOR i = 3 TO 15
          IF i <> 4 and i <> 8 THEN
            res = 0
            IF i AND 1 THEN res = res + (volt_ranges(1) - 1) / Rval
            IF i AND 2 THEN res = res + (volt_ranges(2) - 1) / Rval
            IF i AND 4 THEN res = res + (volt_ranges(4) - 1) / Rval
            IF i AND 8 THEN res = res + (volt_ranges(8) - 1) / Rval
            volt_ranges(i) = (Rval * res + 1)
          ENDIF
        NEXT
        VAR SAVE volt_ranges()
      ELSE
        draw_main_menu
      ENDIF
    EndIf
  ELSEIF tx < MM.HRES/8 THEN
    ' left input/output area touched
  ELSE
    if ty <= MM.VRES/8 THEN
      ' mode bar at top touched
      IF screen_mode$ = "" THEN
        screen_mode$ = "mode_bar"
        draw_mode_bar
      ENDIF
    ELSEIF ty <= MM.VRES / 2 THEN
      IF tx < MM.HRES*9/16 - lfontw*3 THEN
        ' - touched
        touchcmd = "dec_volt"
        inc_dec_voltage -1
        draw_desired
        touchtimer = 100
        touchrepeat = 0
        touchaccel = 1
      ELSEIF tx > MM.HRES*9/16 + lfontw*3 THEN
        ' + touched
        touchcmd = "inc_volt"
        inc_dec_voltage 1
        draw_desired
        touchtimer = 100
        touchrepeat = 0
        touchaccel = 1
      ELSE
        ' voltage/current/resistance touched
        mode_from$ = screen_mode$
        screen_mode$ = "keyboard"
        IF current_mode = 4 THEN
          draw_value_entry "789a456A123<.0cc", "Current"
        ELSEIF current_mode = 5 THEN
          entry_set = 1
          draw_value_entry "789M456O123<.0cc", "Resitance"
        ELSE
          draw_value_entry
        ENDIF
      ENDIF
      ' voltage enter area touched
    ELSEIF ty <= MM.VRES * 3 / 4 THEN
      ' voltage display area touched
    ELSE
      ' info/menu area touched
      IF tx >= MM.HRES*3/4-2 THEN
        draw_main_menu
      ELSE
        output_on = 1 - output_on
        IF output_on > 0 AND output_mode$ = "pulsed" THEN SETTICK output_pulse_ms, output_pulse_off, 2
        draw_screen
      ENDIF
    ENDIF
  ENDIF
END SUB
  
'SUB over_voltage_protection
'  output_on = 0
'  update_relays 0
'  IF screen_mode$ = "" THEN draw_output_state
'  protection_tripped$ = "voltage"
'END SUB
  
  SETPIN BOOST_pin, DOUT
  SETPIN CAL_pin, AIN
  SETPIN REF_pin, AIN
  SETPIN CVFB_pin, AIN
  SETPIN OVP_pin, DIN, PULLUP
'  SETPIN OVP_pin, INTL, over_voltage_protection, PULLUP
  PWM 2, 500000, 50
  init_relays
  mode_changed
  draw_screen
  
  SETPIN 15, INTL, TouchInt
  DO
    update_screen
  LOOP
